<?php

namespace Database\Seeders;

use App\Enums\QuestionType;
use App\Models\Category;
use App\Models\Question;
use Illuminate\Database\Seeder;
use Illuminate\Support\Facades\Storage;

class ReadingListeningQuestionSeeder extends Seeder
{
    private array $readingPassages = [
        "The quick brown fox jumps over the lazy dog. This pangram contains all English letters.",
        "Solar energy is popular renewable source that reduces fossil fuel reliance.",
        "The Great Wall of China stretches over 13,000 miles for protection.",
        "Regular exercise improves health. 30 minutes daily is recommended.",
        "Gutenberg's printing press revolutionized book accessibility."
    ];

    private array $readingQuestions = [
        "What is this pangram's purpose?",
        "What's solar energy's advantage?",
        "How long is the Great Wall?",
        "What's the daily exercise recommendation?",
        "What did the printing press do?"
    ];

    private array $optionContents = [
        "For testing fonts and typing practice",
        "It reduces carbon emissions",
        "Over 13,000 miles",
        "At least 30 minutes of moderate activity",
        "It made books more accessible",
        "For teaching animals tricks",
        "It works at night",
        "About 1,000 miles",
        "One hour of intense exercise",
        "It increased the cost of books",
        "As a children's story",
        "It requires no maintenance",
        "Exactly 5,000 miles",
        "No exercise is needed",
        "It limited access to information",
        "For testing computer keyboards",
        "It only works in sunny countries",
        "Nearly 20,000 miles",
        "15 minutes of stretching",
        "It slowed down communication"
    ];

    public function run(): void
    {
        // Get all reading and listening categories
        $categories = Category::all();

        // Create reading questions
        for ($i = 0; $i < 30; $i++) {
            $question = Question::create([
                'category_id' => $categories->random()->id,
                'question_type' => QuestionType::Reading,
                'question_content' => $this->generateReadingQuestion($i % count($this->readingPassages))
            ]);

            $this->createOptionsForQuestion($question);
        }

        // Create listening questions
        for ($i = 0; $i < 20; $i++) {
            $question = Question::create([
                'category_id' => $categories->random()->id,
                'question_type' => QuestionType::Listening,
                'question_content' => $this->generateListeningAudioPath()
            ]);

            $this->createOptionsForQuestion($question);
        }
    }

    private function generateReadingQuestion(int $index): string
    {
        $passageIndex = $index % count($this->readingPassages);
        $questionIndex = $index % count($this->readingQuestions);

        return "Passage:\n{$this->readingPassages[$passageIndex]}\n\nQuestion: {$this->readingQuestions[$questionIndex]}";
    }

    private function generateListeningAudioPath(): string
    {
        return 'placement-test/4499120250525185653mixkit-crickets-and-insects-in-the-wild-ambience-39.wav';
    }

    private function createOptionsForQuestion(Question $question): void
    {
        $optionsCount = 4;
        $correctOptionIndex = rand(0, $optionsCount - 1);

        for ($i = 0; $i < $optionsCount; $i++) {
            $isCorrect = $i === $correctOptionIndex;
            $optionContent = $this->generateOptionContent($question->question_type, $i);

            $question->options()->create([
                'option_content' => $optionContent,
                'is_correct' => $isCorrect,
            ]);
        }
    }

    private function generateOptionContent(QuestionType $type, int $index): string
    {
        if ($type === QuestionType::Reading) {
            $start = ($index % 5) * 4;
            return $this->optionContents[$start + ($index % 4)] ?? 'Option ' . ($index + 1);
        }

        // For listening questions
        $start = (($index + 2) % 5) * 4;
        return $this->optionContents[$start + (($index + 1) % 4)] ?? 'Choice ' . ($index + 1);
    }
}
